/*
 * Copyright (c) 2005, 2006, Oracle. All rights reserved.  
 */
 
import java.util.logging.LogManager;

import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;

import oracle.jcr.OracleRepository;
import oracle.jcr.OracleRepositoryFactory;

import oracle.jcr.xdb.XDBRepositoryConfiguration;

import oracle.jdbc.pool.OracleDataSource;

/**
 * Traverses the JCR item tree, starting with the specified path.
 */
public class Traverse
{
  /**
   * Entry point from command line.
   * 
   * @param args            command-line arguments
   * 
   * @throws Exception if the operation fails
   */
  public static void main(String[] args)
    throws Exception
  {
    // The XML DB Content Connector uses JDK logging.  To enable logging,
    // comment out the following line.  For further information on JDK
    // logging, refer to:
    //   http://java.sun.com/j2se/1.4.2/docs/guide/util/logging/index.html
    LogManager.getLogManager().reset();

    // Extract the command-line arguments.
    if (args == null || args.length != 4)
    {
      System.out.println(
        "Use: java Traverse <databaseURL> <userName> <password> <absPath>");
        
      System.exit(-1);
    }
    
    String databaseURL = args[0];
    String userName = args[1];
    String password = args[2];
    String absPath = args[3];
    
    // Create a JCR session.
    XDBRepositoryConfiguration repositoryConfiguration =
      new XDBRepositoryConfiguration();
    
    OracleDataSource dataSource = (OracleDataSource)repositoryConfiguration.getDataSource();
    dataSource.setURL(databaseURL);

    OracleRepository repository =
      OracleRepositoryFactory.createOracleRepository(repositoryConfiguration);

    Session session = repository.login(
      new SimpleCredentials(userName, password.toCharArray()), null);
      
    // Get the starting node or property.
    Item item = session.getItem(absPath);
    
    // Perform a depth-first traversal.
    traverse(item, "");
    
    // Disconnect.
    session.logout();
  }
  
  private static void traverse(Item item, String pad)
    throws RepositoryException
  {
    if (item instanceof Property)
    {
      // The item is a property.  Print its name, type, and value.
      Property property = (Property)item;
      String name = property.getName();
      int type = property.getType();
      
      String stringValue = (type == PropertyType.BINARY) ?
        "<binary>" : property.getString();
      
      System.out.println(
        pad + "@" + name + "=" + stringValue +
        " (" + PropertyType.nameFromValue(type) + ")");
    }
    else
    {
      // The item is a node.  Print its name and type.
      Node node = (Node)item;
      String name = node.getName();
      String nodeTypeName = node.getPrimaryNodeType().getName();
      System.out.println(pad + name + " (" + nodeTypeName + ")");
      
      // Process the child properties.
      for (PropertyIterator iter = node.getProperties(); iter.hasNext(); )
      {
        traverse(iter.nextProperty(), pad + "  ");
      }
      
      // Process the child nodes.
      for (NodeIterator iter = node.getNodes(); iter.hasNext(); )
      {
        traverse(iter.nextNode(), pad + "  ");
      }
    }
  }
}

// EOF
